/*
    LinKNX KNX home automation platform
    Copyright (C) 2007 Jean-François Meessen <linknx@ouaye.net>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "objectcontroller.h"
#include "persistentstorage.h"
#include "services.h"
#include <cmath>
extern "C" {
#include "common.h"
}

ObjectController* ObjectController::instance_m;

Object::Object() : gad_m(0), init_m(false), readPending_m(false)
{
}

Object::~Object()
{
}

Object* Object::create(const std::string& type)
{
  if (type == "EIS5")
    return new ValueObject();
  else if (type == "EIS6")
    return new ScalingObject();
  else if (type == "EIS4")
    return new DateObject();
  else if (type == "EIS3")
    return new TimeObject();
  else if (type == "EIS2")
    return new DimmingObject();
  else
    return new SwitchingObject();
}

Object* Object::create(ticpp::Element* pConfig)
{
  std::string type = pConfig->GetAttribute("type");
  Object* obj = Object::create(type);
  if (obj == 0)
  {
		std::stringstream msg;
		msg << "Object type not supported: '" << type << "'" << std::endl;
    throw ticpp::Exception(msg.str());
  }
  obj->importXml(pConfig);
  return obj;
}

void Object::importXml(ticpp::Element* pConfig)
{
  std::string id = pConfig->GetAttribute("id");
  if (id == "")
    throw ticpp::Exception("Missing or empty object ID");
  if (id_m == "")
    id_m = id;

  std::string gad = pConfig->GetAttributeOrDefault("gad", "nochange");
    // set default value to "nochange" just to see if the attribute was present or not in xml
  if (gad == "")
    gad_m = 0;
  else if (gad != "nochange")
    gad_m = readgaddr(gad.c_str());

  forcewrite_m = (pConfig->GetAttribute("forcewrite") == "true");

  initValue_m = pConfig->GetAttribute("init");
  if (initValue_m != "" && initValue_m != "request") {
		std::istringstream val(initValue_m);
		float init_value;
		val >> init_value;
	
		if ( val.fail() )
		{
			if (initValue_m == "persist")
				init_value = PersistentStorage::read(id_m);
			else if (initValue_m == "on" || initValue_m == "true" || initValue_m == "comfort")
				init_value = 1;
			else if (initValue_m == "off" || initValue_m == "false")
				init_value = 0;
			else if (initValue_m == "standby")
				init_value = 2;
			else if (initValue_m == "night")
				init_value = 3;
			else if (initValue_m == "frost")
				init_value = 4;
			else
			{
				std::stringstream msg;
				msg << "Object: Bad init value: '" << initValue_m << "'" << std::endl;
				throw ticpp::Exception(msg.str());
			}
		}
    std::cout << "Object setting initial value to '" << init_value << "'" << std::endl;
    setFloatValue(init_value);
  }

  descr_m = pConfig->GetText(false);
  std::cout << "Configured object '" << id_m << "': gad='" << gad_m << "'" << std::endl;
}

void Object::updateXml(ticpp::Element* pConfig)
{
  std::string gad = pConfig->GetAttributeOrDefault("gad", "nochange");
    // set default value to "nochange" just to see if the attribute was present or not in xml
  if (gad == "")
    gad_m = 0;
  else if (gad != "nochange")
    gad_m = readgaddr(gad.c_str());

  forcewrite_m = (pConfig->GetAttribute("forcewrite") == "true");

  initValue_m = pConfig->GetAttribute("init");
  if (initValue_m != "" && initValue_m != "request") {
		std::istringstream val(initValue_m);
		float init_value;
		val >> init_value;
	
		if ( val.fail() )
		{
			if (initValue_m == "persist")
				init_value = PersistentStorage::read(id_m);
			else if (initValue_m == "on" || initValue_m == "true" || initValue_m == "comfort")
				init_value = 1;
			else if (initValue_m == "off" || initValue_m == "false")
				init_value = 0;
			else if (initValue_m == "standby")
				init_value = 2;
			else if (initValue_m == "night")
				init_value = 3;
			else if (initValue_m == "frost")
				init_value = 4;
			else
			{
				std::stringstream msg;
				msg << "Object: Bad init value: '" << initValue_m << "'" << std::endl;
				throw ticpp::Exception(msg.str());
			}
		}
    std::cout << "Object setting initial value to '" << init_value << "'" << std::endl;
    setFloatValue(init_value);
  }

  descr_m = pConfig->GetText(false);
  std::cout << "Reconfigured object '" << id_m << "': gad='" << gad_m << "'" << std::endl;
}

void Object::exportXml(ticpp::Element* pConfig)
{
    pConfig->SetAttribute("id", id_m);

    if (gad_m != 0)
        pConfig->SetAttribute("gad", writegaddr(gad_m));

    if (initValue_m != "")
        pConfig->SetAttribute("init", initValue_m);

    if (forcewrite_m)
        pConfig->SetAttribute("forcewrite", "true");

    if (descr_m != "")
        pConfig->SetText(descr_m);
}

void Object::read()
{
  KnxConnection* con = Services::instance()->getKnxConnection();
  if (!readPending_m)
  {
    uint8_t buf[2] = { 0, 0 };
    con->write(getGad(), buf, 2);
  }
  readPending_m = true;

	int cnt = 0;
  while (cnt < 100 && readPending_m)
  {
    if (con->isRunning())
      con->checkInput();
    else
      pth_usleep(10000);
    ++cnt;
  }
}

void Object::onUpdate()
{
    ListenerList_t::iterator it;
    for (it = listenerList_m.begin(); it != listenerList_m.end(); it++)
    {
//        std::cout << "Calling onChange on listener for " << id_m << std::endl;
        (*it)->onChange(this);
    }
    if (initValue_m == "persist")
    {
				PersistentStorage::write(id_m, getFloatValue());
    }
}

void Object::onWrite(const uint8_t* buf, int len)
{
    readPending_m = false;
}

void Object::addChangeListener(ChangeListener* listener)
{
    std::cout << "Adding listener to object '" << id_m << "'" << std::endl;
    listenerList_m.push_back(listener);
}
void Object::removeChangeListener(ChangeListener* listener)
{
    listenerList_m.remove(listener);
}

SwitchingObject::SwitchingObject() : value_m(false)
{
}

SwitchingObject::~SwitchingObject()
{
}

void SwitchingObject::onWrite(const uint8_t* buf, int len)
{
    bool newValue;
    Object::onWrite(buf, len);
    if (len == 2)
        newValue = (buf[1] & 0x3F) != 0;
    else
        newValue = buf[2] != 0;
    if (!init_m || newValue != value_m)
    {
        std::cout << "New value " << newValue << " for switching object " << getID() << std::endl;
        value_m = newValue;
        init_m = true;
        onUpdate();
    }
}

void SwitchingObject::setBoolValue(bool value)
{
    if (!init_m || value != value_m || forcewrite_m)
    {
        value_m = value;
        uint8_t buf[3] = { 0, 0x80 };
        buf[1] = value ? 0x81 : 0x80;
        Services::instance()->getKnxConnection()->write(getGad(), buf, 2);
        init_m = true;
        onUpdate();
    }
}

DimmingObject::DimmingObject() : value_m(0)
{
}

DimmingObject::~DimmingObject()
{
}

void DimmingObject::exportXml(ticpp::Element* pConfig)
{
    Object::exportXml(pConfig);
    pConfig->SetAttribute("type", "EIS2");
}

void DimmingObject::onWrite(const uint8_t* buf, int len)
{
    int newValue;
    Object::onWrite(buf, len);
    if (len == 2)
        newValue = (buf[1] & 0x3F);
    else
        newValue = buf[2];

    if (!init_m || newValue != value_m)
    {
        std::cout << "New value " << newValue << " for dimming object " << getID() << std::endl;
        value_m = newValue;
        init_m = true;
        onUpdate();
    }
}

void DimmingObject::setIntValue(int value)
{
    if (!init_m || value != value_m || forcewrite_m)
    {
        value_m = value;
        uint8_t buf[3] = { 0, 0x80 };
        buf[1] = (value & 0x0f) | 0x80;
        Services::instance()->getKnxConnection()->write(getGad(), buf, 2);

        init_m = true;
        onUpdate();
    }
}

TimeObject::TimeObject() : wday_m(0), hour_m(0), min_m(0), sec_m(0)
{
}

TimeObject::~TimeObject()
{
}

void TimeObject::exportXml(ticpp::Element* pConfig)
{
    Object::exportXml(pConfig);
    pConfig->SetAttribute("type", "EIS3");
}

void TimeObject::onWrite(const uint8_t* buf, int len)
{
    if (len < 5)
    {
        std::cout << "Invlalid packet received for TimeObject (too short)" << std::endl;
        return;
    }
    int wday, hour, min, sec;
    Object::onWrite(buf, len);

    wday = (buf[2] & 0xE0) >> 5;
    hour = buf[2] & 0x1F;
    min = buf[3];
    sec = buf[4];
    if (!init_m || wday != wday_m || hour != hour_m || min != min_m || sec != sec_m)
    {
        std::cout << "New value " << wday << " " << hour << ":" << min << ":" << sec << " for time object " << getID() << std::endl;
        wday_m = wday;
        hour_m = hour;
        min_m = min;
        sec_m = sec;
        init_m = true;
        onUpdate();
    }
}

void TimeObject::setIntValue(int value)
{
    if (value < 0)
        setTime(time(0));
    else
    {
        int rem = value;
        int sec = rem % 60;
        rem = (rem-sec) / 60; 
        int min = rem % 60;
        rem = (rem-min) / 60; 
        int hour = rem % 24;
        rem = (rem-hour) / 24; 
        int wday = rem % 8;

        setTime(wday, hour, min, sec);
    }
}

int TimeObject::getIntValue()
{
    if (!init_m)
        read();
    return ((wday_m*24 + hour_m) * 60 + min_m) * 60 + sec_m;
}

void TimeObject::setTime(time_t time)
{
    struct tm * timeinfo = localtime(&time);
    int wday = timeinfo->tm_wday;
    if (wday == 0)
        wday = 7;
		setTime(wday, timeinfo->tm_hour, timeinfo->tm_min, timeinfo->tm_sec);
}

void TimeObject::setTime(int wday, int hour, int min, int sec)
{
    if (!init_m ||
        wday_m != wday ||
        hour_m != hour ||
        min_m != min ||
        sec_m != sec ||
        forcewrite_m)
    {
        std::cout << "TimeObject: setTime "
						    	<< wday << " "
							    << hour << ":"
							    << min << ":"
							    << sec << std::endl;
        wday_m = wday;
        hour_m = hour;
        min_m = min;
        sec_m = sec;

        uint8_t buf[5] = { 0, 0x80, ((wday<<5) & 0xE0) | (hour & 0x1F), min, sec };
        
        Services::instance()->getKnxConnection()->write(getGad(), buf, 5);
        init_m = true;
        onUpdate();
    }
}

DateObject::DateObject() : day_m(0), month_m(0), year_m(0)
{
}

DateObject::~DateObject()
{
}

void DateObject::exportXml(ticpp::Element* pConfig)
{
    Object::exportXml(pConfig);
    pConfig->SetAttribute("type", "EIS4");
}

void DateObject::onWrite(const uint8_t* buf, int len)
{
    if (len < 5)
    {
        std::cout << "Invlalid packet received for DateObject (too short)" << std::endl;
        return;
    }
    int day, month, year;
    Object::onWrite(buf, len);

    day = buf[2];
    month = buf[3];
    year = buf[4];
    if (year < 90)
        year += 100;
    if (!init_m || day != day_m || month != month_m || year != year_m)
    {
        std::cout << "New value " << year+1900 << "-" << month << "-" << day << " for date object " << getID() << std::endl;
        day_m = day;
        month_m = month;
        year_m = year;
        init_m = true;
        onUpdate();
    }
}

void DateObject::setIntValue(int value)
{
    if (value < 0)
        setDate(time(0));
    else
    {
        int rem = value;
        int day = rem % 32;
        rem = (rem-day) / 32; 
        int month = rem % 13;
        rem = (rem-month) / 13; 
        int year = rem % 256;

        setDate(day, month, year);
    }
}

int DateObject::getIntValue()
{
    if (!init_m)
        read();
    return (year_m*13 + month_m) * 32 + day_m;
}

void DateObject::setDate(time_t time)
{
    struct tm * timeinfo = localtime(&time);
		setDate(timeinfo->tm_mday, timeinfo->tm_mon+1, timeinfo->tm_year);
}

void DateObject::setDate(int day, int month, int year)
{
    if (year >= 1900)
        year -= 1900;
    if (!init_m ||
        day_m != day ||
        month_m != month ||
        year_m != year ||
        forcewrite_m)
    {
        std::cout << "DateObject: setDate "
						    	<< year + 1900 << "-"
							    << month << "-"
							    << day << std::endl;
        day_m = day;
        month_m = month;
        year_m = year;

        uint8_t buf[5] = { 0, 0x80, day, month, year };
        
        Services::instance()->getKnxConnection()->write(getGad(), buf, 5);
        init_m = true;
        onUpdate();
    }
}

ValueObject::ValueObject() : value_m(0)
{
}

ValueObject::~ValueObject()
{
}

void ValueObject::exportXml(ticpp::Element* pConfig)
{
    Object::exportXml(pConfig);
    pConfig->SetAttribute("type", "EIS5");
}

void ValueObject::onWrite(const uint8_t* buf, int len)
{
    if (len < 4)
    {
        std::cout << "Invlalid packet received for ValueObject (too short)" << std::endl;
        return;
    }
    float newValue;
    Object::onWrite(buf, len);
    int d1 = ((unsigned char) buf[2]) * 256 + (unsigned char) buf[3];
    int m = d1 & 0x7ff;
    if (d1 & 0x8000)
        m |= ~0x7ff;
    int ex = (d1 & 0x7800) >> 11;
    newValue = ((float)m * (1 << ex) / 100);
//		printf ("d1=%d;m=%d;ex=%d;temp=%f\n", d1, m, ex, temp);
    if (!init_m || newValue != value_m)
    {
        std::cout << "New value " << newValue << " for value object " << getID() << std::endl;
        value_m = newValue;
        init_m = true;
        onUpdate();
    }
}

void ValueObject::setFloatValue(float value)
{
    if (!init_m || value != value_m || forcewrite_m)
    {
        value_m = value;
        uint8_t buf[4] = { 0, 0x80, 0, 0 };
        int ex = 0;
        int m = (int)rint(value * 100);
        if (m < 0)
        {
            m = -m;
            while (m > 2048)
            {
                m = m >> 1;
                ex++;
            }
            m = -m;
            buf[2] = ((m >> 8) & 0x07) | ((ex << 3) & 0x78) | (1 << 7);
        }
        else
        {
            while (m > 2047)
            {
                m = m >> 1;
                ex++;
            }
            buf[2] = ((m >> 8) & 0x07) | ((ex << 3) & 0x78);
        }
        buf[3] = (m & 0xff);
        
        Services::instance()->getKnxConnection()->write(getGad(), buf, 4);
        init_m = true;
        onUpdate();
    }
}

ScalingObject::ScalingObject() : value_m(0)
{
}

ScalingObject::~ScalingObject()
{
}

void ScalingObject::exportXml(ticpp::Element* pConfig)
{
    Object::exportXml(pConfig);
    pConfig->SetAttribute("type", "EIS6");
}

void ScalingObject::onWrite(const uint8_t* buf, int len)
{
    int newValue;
    Object::onWrite(buf, len);
    if (len == 2)
        newValue = (buf[1] & 0x3F);
    else
        newValue = buf[2];
    if (!init_m || newValue != value_m)
    {
        std::cout << "New value " << newValue << " for scaling object " << getID() << std::endl;
        value_m = newValue;
        init_m = true;
        onUpdate();
    }
}

void ScalingObject::setIntValue(int value)
{
    if (!init_m || value != value_m || forcewrite_m)
    {
        value_m = value;
        uint8_t buf[3] = { 0, 0x80, 0 };
        buf[2] = (value & 0xff);
        
        Services::instance()->getKnxConnection()->write(getGad(), buf, 3);
        init_m = true;
        onUpdate();
    }
}

ObjectController::ObjectController()
{
    Services::instance()->getKnxConnection()->addTelegramListener(this);
}

ObjectController::~ObjectController()
{
  ObjectIdMap_t::iterator it;
  for (it = objectIdMap_m.begin(); it != objectIdMap_m.end(); it++)
    delete (*it).second;
}

ObjectController* ObjectController::instance()
{
  if (instance_m == 0)
    instance_m = new ObjectController();
  return instance_m;
}

void ObjectController::onWrite(eibaddr_t src, eibaddr_t dest, const uint8_t* buf, int len)
{
        ObjectMap_t::iterator it = objectMap_m.find(dest);
        if (it != objectMap_m.end())
          (*it).second->onWrite(buf, len);
}

void ObjectController::onRead(eibaddr_t src, eibaddr_t dest, const uint8_t* buf, int len) { };
void ObjectController::onResponse(eibaddr_t src, eibaddr_t dest, const uint8_t* buf, int len) { onWrite(src, dest, buf, len); };

Object* ObjectController::getObject(const std::string& id)
{
  ObjectIdMap_t::iterator it = objectIdMap_m.find(id);
  if (it == objectIdMap_m.end())
  {
		std::stringstream msg;
		msg << "ObjectController: Object ID not found: '" << id << "'" << std::endl;
		throw ticpp::Exception(msg.str());
  }
  return (*it).second;
}

void ObjectController::addObject(Object* object)
{
  if (!objectIdMap_m.insert(ObjectIdPair_t(object->getID(), object)).second)
    throw ticpp::Exception("Object ID already exists");
  if (object->getGad() && !objectMap_m.insert(ObjectPair_t(object->getGad(), object)).second)
    throw ticpp::Exception("Object GAD is already registered");
}

void ObjectController::importXml(ticpp::Element* pConfig)
{
  ticpp::Iterator< ticpp::Element > child("object");
  for ( child = pConfig->FirstChildElement("object", false); child != child.end(); child++ )
  {
    std::string id = child->GetAttribute("id");
    bool del = child->GetAttribute("delete") == "true";
    ObjectIdMap_t::iterator it = objectIdMap_m.find(id);
    if (it == objectIdMap_m.end())
    {
      if (del)
        throw ticpp::Exception("Object not found");
      Object* object = Object::create(&(*child));
      if (object->getGad() && !objectMap_m.insert(ObjectPair_t(object->getGad(), object)).second)
      {
        delete object;
        throw ticpp::Exception("Object GAD is already registered");
      }
      objectIdMap_m.insert(ObjectIdPair_t(id, object));
    }
    else if (del)
    {
      eibaddr_t gad = it->second->getGad();
      if (gad)
        objectMap_m.erase(gad);
      delete it->second;
      objectIdMap_m.erase(it);
    }
    else
    {
      eibaddr_t gad = it->second->getGad();
      it->second->importXml(&(*child));
      eibaddr_t gad2 = it->second->getGad();
      if (gad != gad2)
      {
        if (gad2)
        {
          if (!objectMap_m.insert(ObjectPair_t(gad2, it->second)).second)
            throw ticpp::Exception("New object GAD is already registered");
        }
        if (gad)
          objectMap_m.erase(gad);
      }
    }
  }

}

void ObjectController::exportXml(ticpp::Element* pConfig)
{
  ObjectIdMap_t::iterator it;
  for (it = objectIdMap_m.begin(); it != objectIdMap_m.end(); it++)
  {
    ticpp::Element pElem("object");
    (*it).second->exportXml(&pElem);
    pConfig->LinkEndChild(&pElem);
  }
}
