/*
    EIBD eib bus access and management daemon
    Copyright (C) 2005-2006 Martin K�gler <mkoegler@auto.tuwien.ac.at>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "objectcontroller.h"

ObjectController* ObjectController::instance_m;

Object::Object() : gad_m(0), persist_m(false)
{
}

Object::~Object()
{
}

Object* Object::create(const std::string& type)
{
  if (type == "EIS5")
    return new ValueObject();
  else if (type == "EIS6")
    return new ScalingObject();
  else
    return new SwitchingObject();
}

Object* Object::create(ticpp::Element* pConfig)
{
  std::string type = pConfig->GetAttribute("type");
  Object* obj = Object::create(type);
  if (obj == 0)
  {
		std::stringstream msg;
		msg << "Object type not supported: '" << type << "'" << std::endl;
    throw ticpp::Exception(msg.str());
  }
  obj->importXml(pConfig);
  return obj;
}

void Object::importXml(ticpp::Element* pConfig)
{
  id_m = pConfig->GetAttribute("id");
  std::string gad = pConfig->GetAttribute("gad");
  if (id_m == "")
    throw ticpp::Exception("Missing or empty object ID");
  ObjectController::instance()->addObject(this);

  if (gad != "")
    gad_m = readgaddr(gad.c_str());
  persist_m = (pConfig->GetAttribute("persist") == "true");
  descr_m = pConfig->GetText(false);
  std::cout << "Configured object '" << id_m << "': gad='" << gad_m << "' persist='" << persist_m << "'" << std::endl;
}

void Object::exportXml(ticpp::Element* pConfig)
{
}

void Object::onUpdate()
{
    ListenerList_t::iterator it;
    for (it = listenerList_m.begin(); it != listenerList_m.end(); it++)
    {
//        std::cout << "Calling onChange on listener for " << id_m << std::endl;
        (*it)->onChange(this);
    }
}

void Object::addChangeListener(ChangeListener* listener)
{
    std::cout << "Adding listener to object '" << id_m << "'" << std::endl;
    listenerList_m.push_back(listener);
}
void Object::removeChangeListener(ChangeListener* listener)
{
    listenerList_m.remove(listener);
}

SwitchingObject::SwitchingObject() : value_m(false)
{
}

SwitchingObject::~SwitchingObject()
{
}

void SwitchingObject::onWrite(const uchar* buf, int len)
{
    bool newValue;
    if (len == 2)
        newValue = (buf[1] & 0x3F) != 0;
    else
        newValue = buf[2] != 0;
    if (newValue != value_m)
    {
        std::cout << "New value " << newValue << " for switching object " << getGad() << std::endl;
        value_m = newValue;
        onUpdate();
    }
}

void SwitchingObject::setIntValue(int value)
{
    setBoolValue(value != 0);
}

void SwitchingObject::setFloatValue(float value)
{
    setBoolValue(value != 0);
}

void SwitchingObject::setBoolValue(bool value)
{
    if (value != value_m)
    {
        value_m = value;
        uint8_t buf[3] = { 0, 0x80 };
        buf[1] = value ? 0x81 : 0x80;
        ObjectController::instance()->write(getGad(), buf, 2);
        onUpdate();
    }
}


ValueObject::ValueObject() : value_m(0)
{
}

ValueObject::~ValueObject()
{
}

void ValueObject::onWrite(const uchar* buf, int len)
{
    if (len < 4)
    {
        std::cout << "Invlalid packet received for ValueObject (too short)" << std::endl;
        return;
    }
    float newValue;
    int d1 = ((unsigned char) buf[2]) * 256 + (unsigned char) buf[3];
    int m = d1 & 0x7ff;
    int ex = (d1 & 0x7800) >> 11;
    newValue = ((float)m * (1 << ex) / 100);
//		printf ("d1=%d;m=%d;ex=%d;temp=%f\n", d1, m, ex, temp);
    if (newValue != value_m)
    {
        std::cout << "New value " << newValue << " for value object " << getGad() << std::endl;
        value_m = newValue;
        onUpdate();
    }
}

void ValueObject::setIntValue(int value)
{
    setFloatValue((float)value);
}

void ValueObject::setFloatValue(float value)
{
    if (value != value_m)
    {
        value_m = value;
        uint8_t buf[4] = { 0, 0x80, 0, 0 };
        int ex = 0;
        int m = (int)(value * 100);
        while (m > 2047 || m < -2048)
        {
            m >> 1;
            ex++;
        }
        if (m < 0)
        {
            m += 2048;
            buf[2] = ((m >> 8) & 0x07) | ((ex << 3) & 0x78) | (1 << 7);
        }
        else
        {
            buf[2] = ((m >> 8) & 0x07) | ((ex << 3) & 0x78);
        }
        buf[3] = (m & 0xff);
        
        ObjectController::instance()->write(getGad(), buf, 4);
        onUpdate();
    }
}

void ValueObject::setBoolValue(bool value)
{
    setFloatValue(value ? 1 : 0);
}



ScalingObject::ScalingObject() : value_m(0)
{
}

ScalingObject::~ScalingObject()
{
}

void ScalingObject::onWrite(const uchar* buf, int len)
{
    int newValue;
    if (len == 2)
        newValue = (buf[1] & 0x3F);
    else
        newValue = buf[2];
    if (newValue != value_m)
    {
        std::cout << "New value " << newValue << " for scaling object " << getGad() << std::endl;
        value_m = newValue;
        onUpdate();
    }
}

void ScalingObject::setIntValue(int value)
{
    if (value != value_m)
    {
        value_m = value;
        uint8_t buf[3] = { 0, 0x80, 0 };
        buf[2] = (value & 0xff);
        
        ObjectController::instance()->write(getGad(), buf, 3);
        onUpdate();
    }
}

void ScalingObject::setFloatValue(float value)
{
    setIntValue((int)value);
}

void ScalingObject::setBoolValue(bool value)
{
    setIntValue(value ? 1 : 0);
}


ObjectController::ObjectController()
{
}

ObjectController::~ObjectController()
{
  Stop ();
}

ObjectController* ObjectController::instance()
{
  if (instance_m == 0)
    instance_m = new ObjectController();
  return instance_m;
}

void ObjectController::Run (pth_sem_t * stop1)
{
  int len;
  eibaddr_t dest;
  eibaddr_t src;
  uchar buf[200];

  if (EIBOpen_GroupSocket (con_m, 0) == -1)
    die ("Connect failed");
  pth_event_t stop = pth_event (PTH_EVENT_SEM, stop1);
  while (pth_event_status (stop) != PTH_STATUS_OCCURRED)
  {
    len = EIBGetGroup_Src (con_m, sizeof (buf), buf, &src, &dest);
    if (len == -1)
      die ("Read failed");
    if (len < 2)
      die ("Invalid Packet");
    if (buf[0] & 0x3 || (buf[1] & 0xC0) == 0xC0)
    {
      printf ("Unknown APDU from ");
      printIndividual (src);
      printf (" to ");
      printGroup (dest);
      printf (": ");
      printHex (len, buf);
      printf ("\n");
    }
    else
    {
      switch (buf[1] & 0xC0)
      {
      case 0x00:
        printf ("Read");
        break;
      case 0x40:
        printf ("Response");
        break;
      case 0x80:
        printf ("Write");
        break;
      }
      printf (" from ");
      printIndividual (src);
      printf (" to ");
      printGroup (dest);
      if (buf[1] & 0xC0)
      {
        printf (": ");
        if (len == 2)
          printf ("%02X", buf[1] & 0x3F);
        else
          printHex (len - 2, buf + 2);
      }
      printf ("\n");
      if (buf[1] & 0x80)
      {
        ObjectMap_t::iterator it = objectMap_m.find(dest);
        if (it != objectMap_m.end())
          (*it).second->onWrite(buf, len);
      }
    }
    
  }
  pth_event_free (stop, PTH_FREE_THIS);
}

Object* ObjectController::getObject(const std::string& id)
{
  ObjectIdMap_t::iterator it = objectIdMap_m.find(id);
  if (it == objectIdMap_m.end())
    throw ticpp::Exception("Object ID not found");
  return (*it).second;
}

void ObjectController::addObject(Object* object)
{
  if (!objectIdMap_m.insert(ObjectIdPair_t(object->getID(), object)).second)
    throw ticpp::Exception("Object ID already exists");
  if (object->getGad() && !objectMap_m.insert(ObjectPair_t(object->getGad(), object)).second)
    throw ticpp::Exception("Object GAD is already registered");
}

void ObjectController::importXml(ticpp::Element* pConfig)
{
  ticpp::Iterator< ticpp::Element > child;
  for ( child = pConfig->FirstChildElement(); child != child.end(); child++ )
  {
    if (child->Value() == "knxConnection")
    {
      std::string url = child->GetAttribute("url");
      con_m = EIBSocketURL(url.c_str());
      if (!con_m)
        throw ticpp::Exception("Failed to open knxConnection url");
    }
    else if (child->Value() == "object")
    {
      Object* object = Object::create(&(*child));
      objectIdMap_m[object->getID()] = object;
      eibaddr_t gad = object->getGad();
      if (gad)
        objectMap_m[gad] = object;
    }
  }

}

void ObjectController::exportXml(ticpp::Element* pConfig)
{
}

void ObjectController::startServer()
{
  Start();
}

void ObjectController::stopServer()
{
  Stop();
}

void ObjectController::write(eibaddr_t gad, uint8_t* buf, int len)
{
  if(gad == 0)
    return;
  printf ("ObjectController::write(gad=%d, buf, len=%d):", gad, len);
	printHex (len, buf);
  printf ("\n");
  if (con_m)
  {
    len = EIBSendGroup (con_m, gad, len, buf);
    if (len == -1)
      die ("Request failed");
    printf ("Send request\n");
  }
}
