/*
    LinKNX KNX home automation platform
    Copyright (C) 2007 Jean-François Meessen <linknx@ouaye.net>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifndef OBJECTCONTROLLER_H
#define OBJECTCONTROLLER_H

#include <list>
#include <string>
#include <map>
#include "config.h"
#include "threads.h"
extern "C" {
#include "common.h"
}
#include "ticpp.h"
#include "eibclient.h"

class Object;

class ChangeListener
{
public:
    virtual void onChange(Object* object) = 0;
};

class Object
{
public:
    Object();
    virtual ~Object();

    static Object* create(ticpp::Element* pConfig);
    static Object* create(const std::string& type);

    virtual void importXml(ticpp::Element* pConfig);
    virtual void exportXml(ticpp::Element* pConfig);

    const char* getID() { return id_m.c_str(); };
    const char* getDescr() { return descr_m.c_str(); };
    const eibaddr_t getGad() { return gad_m; };
    void read();
    virtual void onUpdate();
    void addChangeListener(ChangeListener* listener);
    void removeChangeListener(ChangeListener* listener);
    virtual void onWrite(const uchar* buf, int len) {};
    virtual void setIntValue(int value) {};
    virtual void setFloatValue(float value) {};
    virtual void setBoolValue(bool value) {};
    virtual int getIntValue() { return getBoolValue() ? 1 : 0; };
    virtual float getFloatValue() { return getBoolValue() ? 1 : 0; };
    virtual bool getBoolValue() { return false; };
protected:
    bool init_m;
private:
    std::string id_m;
    eibaddr_t gad_m;
    std::string initValue_m;
    std::string descr_m;
    typedef std::list<ChangeListener*> ListenerList_t;
    ListenerList_t listenerList_m;
};

class SwitchingObject : public Object
{
public:
    SwitchingObject();
    virtual ~SwitchingObject();

    virtual void onWrite(const uchar* buf, int len);
    virtual void setIntValue(int value);
    virtual void setFloatValue(float value);
    virtual void setBoolValue(bool value);
    virtual bool getBoolValue() { if (!init_m) read(); return value_m; };
private:
    bool value_m;
};

class ValueObject : public Object
{
public:
    ValueObject();
    virtual ~ValueObject();

    virtual void exportXml(ticpp::Element* pConfig);

    virtual void onWrite(const uchar* buf, int len);
    virtual void setIntValue(int value);
    virtual void setFloatValue(float value);
    virtual void setBoolValue(bool value);
    virtual int getIntValue() { return (int)getFloatValue(); };
    virtual float getFloatValue() { if (!init_m) read(); return value_m; };
    virtual bool getBoolValue() { return getFloatValue() != 0; };
private:
    float value_m;
};

class ScalingObject : public Object
{
public:
    ScalingObject();
    virtual ~ScalingObject();

    virtual void exportXml(ticpp::Element* pConfig);

    virtual void onWrite(const uchar* buf, int len);
    virtual void setIntValue(int value);
    virtual void setFloatValue(float value);
    virtual void setBoolValue(bool value);
    virtual int getIntValue() { if (!init_m) read(); return value_m; };
    virtual float getFloatValue() { return getIntValue(); };
    virtual bool getBoolValue() { return getIntValue() != 0; };
private:
    int value_m;
};

class ObjectController : protected Thread
{
public:
    static ObjectController* instance();
    void addObject(Object* object);
    void removeObject(Object* object);

    Object* getObject(const std::string& id);
    void write(eibaddr_t gad, uint8_t* buf, int len);

    virtual void importXml(ticpp::Element* pConfig);
    virtual void exportXml(ticpp::Element* pConfig);
    void startServer();
    void stopServer();

private:
    ObjectController();
    virtual ~ObjectController();

    void Run (pth_sem_t * stop);

    typedef std::pair<eibaddr_t ,Object*> ObjectPair_t;
    typedef std::map<eibaddr_t ,Object*> ObjectMap_t;
    typedef std::pair<std::string ,Object*> ObjectIdPair_t;
    typedef std::map<std::string ,Object*> ObjectIdMap_t;
    ObjectMap_t objectMap_m;
    ObjectIdMap_t objectIdMap_m;
    static ObjectController* instance_m;
    EIBConnection *con_m;
    std::string url_m;
};

#endif
