/*
    EIBD eib bus access and management daemon
    Copyright (C) 2005-2006 Martin K�gler <mkoegler@auto.tuwien.ac.at>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <argp.h>
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <signal.h>
#include <unistd.h>
#include <fcntl.h>
#include <cstring>
#include <pthsem.h>
#define TIXML_USE_TICPP
#include "ticpp.h"
#include "eibclient.h"
#include "objectcontroller.h"
#include "ruleserver.h"

/** structure to store the arguments */
struct arguments
{
  /** path for config file */
  const char *configfile;
  /** path to pid file */
  const char *pidfile;
  /** path to trace log file */
  const char *daemon;
};
/** storage for the arguments*/
struct arguments arg;

/** aborts program with a printf like message */
void
die (const char *msg, ...)
{
  va_list ap;
  va_start (ap, msg);
  vprintf (msg, ap);
  printf ("\n");
  va_end (ap);

  if (arg.pidfile)
    unlink (arg.pidfile);

  exit (1);
}


/** parses an EIB individual address */
eibaddr_t
readaddr (const char *addr)
{
  int a, b, c;
  sscanf (addr, "%d.%d.%d", &a, &b, &c);
  return ((a & 0x0f) << 12) | ((b & 0x0f) << 8) | ((c & 0xff));
}

/** version */
const char *argp_program_version = "linknx 0.1";
/** documentation */
static char doc[] =
  "linknx -- an automation server for EIB/KNX\n"
  "(C) 2007 Jean-François Meessen <jf.meessen (at) ouaye.net>\n";

/** documentation for arguments*/
static char args_doc[] = "URL";

/** option list */
static struct argp_option options[] = {
  {"config", 'c', "FILE", OPTION_ARG_OPTIONAL,
   "read configuration from file (default /tmp/linknx.xml)"},
  {"pid-file", 'p', "FILE", 0, "write the PID of the process to FILE"},
  {"daemon", 'd', "FILE", OPTION_ARG_OPTIONAL,
   "start the programm as daemon, the output will be written to FILE, if the argument present"},
  {0}
};

/** parses and stores an option */
static error_t
parse_opt (int key, char *arg, struct argp_state *state)
{
  struct arguments *arguments = (struct arguments *) state->input;
  switch (key)
    {
    case 'c':
      arguments->configfile = (char *) (arg ? arg : "/tmp/linknx.xml");
      break;
    case 'p':
      arguments->pidfile = arg;
      break;
    case 'd':
      arguments->daemon = (char *) (arg ? arg : "/dev/null");
      break;
    default:
      return ARGP_ERR_UNKNOWN;
    }
  return 0;
}

/** information for the argument parser*/
static struct argp argp = { options, parse_opt, args_doc, doc };

int
main (int ac, char *ag[])
{
  int index;

  memset (&arg, 0, sizeof (arg));

  argp_parse (&argp, ac, ag, 0, &index, &arg);
  if (index > ac - 1)
    die ("url expected");
  if (index < ac - 1)
    die ("unexpected parameter");

  signal (SIGPIPE, SIG_IGN);
  pth_init ();

  if (arg.daemon)
    {
      int fd = open (arg.daemon, O_WRONLY | O_APPEND | O_CREAT);
      if (fd == -1)
	die ("Can not open file %s", arg.daemon);
      int i = fork ();
      if (i < 0)
	die ("fork failed");
      if (i > 0)
	exit (0);
      close (1);
      close (2);
      close (0);
      dup2 (fd, 1);
      dup2 (fd, 2);
      setsid ();
    }


  FILE *pidf;
  if (arg.pidfile)
    if ((pidf = fopen (arg.pidfile, "w")) != NULL)
      {
	fprintf (pidf, "%d", getpid ());
	fclose (pidf);
      }

  RuleServer rules;
  ObjectController* objects = ObjectController::instance();
  try
  {
  	// Load a document
	  ticpp::Document doc(arg.configfile);
    doc.LoadFile();

    ticpp::Element* pConfig = doc.FirstChildElement("config");
    ticpp::Element* pRules = pConfig->FirstChildElement("rules");
    rules.importXml(pRules);
    ticpp::Element* pObjects = pConfig->FirstChildElement("objects");
    objects->importXml(pObjects);
  }
  catch( ticpp::Exception& ex )
  {
    // If any function has an error, execution will enter here.
    // Report the error
    std::cout << ex.m_details;
    die ("initialisation failed");
  }

  sigset_t t1;
  sigemptyset (&t1);
  sigaddset (&t1, SIGINT);
  sigaddset (&t1, SIGTERM);
  signal (SIGINT, SIG_IGN);
  signal (SIGTERM, SIG_IGN);

  int x;
  pth_sigwait (&t1, &x);

  signal (SIGINT, SIG_DFL);
  signal (SIGTERM, SIG_DFL);

  if (arg.pidfile)
    unlink (arg.pidfile);

  pth_exit (0);
  return 0;
}
